<?php

namespace App\Http\Controllers;

use App\Contact;
use App\EmptyProductReturn;
use App\Product;
use App\ReturnableProductStock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;

class EmptyProductReturnController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (request()->ajax()) {
            return DataTables::of(EmptyProductReturn::query())
                ->addColumn(
                    'action',
                    '   <button data-href="{{action(\'App\Http\Controllers\EmptyProductReturnController@edit\', [$id])}}" class="btn btn-xs btn-primary edit_empty_product_return_button"><i class="glyphicon glyphicon-edit"></i> @lang("messages.edit")</button>
                        &nbsp;
                        <button data-href="{{action(\'App\Http\Controllers\EmptyProductReturnController@destroy\', [$id])}}" class="btn btn-xs btn-danger delete_empty_product_return_button"><i class="glyphicon glyphicon-trash"></i> @lang("messages.delete")</button>'
                )
                ->editColumn('contact_id', function($data) {
                    return $data->contact?->name ?? '';
                })
                ->editColumn('product_id', function($data) {
                    return $data->product?->name;
                })
                ->editColumn('created_at', function($data) {
                    return $data->created_at->format('d-m-Y g:i A');
                })
                ->removeColumn('id')
                ->removeColumn('updated_at')
                ->rawColumns([4])
                ->make(false);
        }
        return view('empty-product-return.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $products = Product::forDropdown(1, true);
        $contacts = Contact::contactDropdown(1, true);
        return view('empty-product-return.create', compact('products', 'contacts'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'product_id' => ['required','integer','exists:products,id'],
            'contact_id' => ['required','integer','exists:contacts,id'],
            'quantity'   => ['required','integer'],
        ]);

        try {
            $stock = ReturnableProductStock::query()->where('product_id', $request->product_id)->first();
            
            if($stock && $stock->quantity >= $request->quantity) {
                $return = EmptyProductReturn::query()->create($validated);

                $stock->decrement('quantity', $request->quantity);

                $output = [
                    'success' => true,
                    'data'    => $return,
                    'msg'     => "Product stock added succesfully",
                ];
            }
            else {
                $output = [
                    'success' => false,
                    'msg' => "Not enough stock available",
                ];
            }
            
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'Something went wrong, please try again',
            ];
        }
        return $output;
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(EmptyProductReturn $emptyProductReturn)
    {
        $products = Product::forDropdown(1, true);
        $contacts = Contact::contactDropdown(1, true);
        return view('empty-product-return.edit', compact('products', 'contacts', 'emptyProductReturn'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, EmptyProductReturn $emptyProductReturn)
    {
        $validated = $request->validate([
            'product_id' => ['required','integer','exists:products,id'],
            'contact_id' => ['required','integer','exists:contacts,id'],
            'quantity'   => ['required','integer'],
        ]);

        try {
            $stock = ReturnableProductStock::query()->where('product_id', $request->product_id)->first();

            if($stock && ($stock->quantity + $emptyProductReturn->quantity) >= $request->quantity) {
                if($emptyProductReturn->quantity > $request->quantity) {
                    $stock->increment('quantity', ($emptyProductReturn->quantity - $request->quantity));
                }
                else {
                    $stock->decrement('quantity', ($request->quantity - $emptyProductReturn->quantity));
                }

                $emptyProductReturn->update($validated);
                $output = [
                    'success' => true,
                    'data'    => $emptyProductReturn,
                    'msg'     => "Product stock added succesfully",
                ];
            }
            else {
                $output = [
                    'success' => false,
                    'data'    => $emptyProductReturn,
                    'msg' => "Not enough stock available",
                ];
            }
            
        } catch (\Exception $e) {
            Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = [
                'success' => false,
                'msg' => 'Something went wrong, please try again',
            ];
        }
        return $output;
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(EmptyProductReturn $emptyProductReturn)
    {
        if (request()->ajax()) {
            try {
                $stock = ReturnableProductStock::query()->where('product_id', $emptyProductReturn->product_id)->first();
                if($stock) {
                    $stock->increment('quantity', $emptyProductReturn->quantity);
                }
                $emptyProductReturn->delete();
                $output = [
                    'success' => true,
                    'msg' => 'Product due deleted succesfully',
                ];
            } catch (\Exception $e) {
                Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

                $output = [
                    'success' => false,
                    'msg' => 'Something went wrong, please try again',
                ];
            }

            return $output;
        }
    }
}
